<?php
declare(strict_types=1);

error_reporting(E_ALL);
ini_set('display_errors', '1');

if (!function_exists('str_contains')) {
    function str_contains(string $haystack, string $needle): bool
    {
        return $needle === '' || strpos($haystack, $needle) !== false;
    }
}

if (!function_exists('str_starts_with')) {
    function str_starts_with(string $haystack, string $needle): bool
    {
        if ($needle === '') {
            return true;
        }
        return strncmp($haystack, $needle, strlen($needle)) === 0;
    }
}

$segments = getPathSegments();
$firstSegment = $segments[0] ?? null;

if ($firstSegment === 'install') {
    serveInstaller();
}

if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    sendCorsHeaders();
    http_response_code(204);
    exit;
}

$configFile = __DIR__ . '/config.php';

if (!file_exists($configFile)) {
    redirectToInstall();
}

$config = require $configFile;

if (!isset($config['db'])) {
    respondJson(500, [
        'success' => false,
        'error'   => 'invalid_config',
        'message' => 'El archivo config.php no contiene la sección db.',
    ]);
}

$db = $config['db'];

try {
    $pdo = new PDO(
        sprintf(
            'mysql:host=%s;port=%s;dbname=%s;charset=%s',
            $db['host'],
            $db['port'],
            $db['name'],
            $db['charset'] ?? 'utf8mb4'
        ),
        $db['user'],
        $db['pass'],
        [
            PDO::ATTR_ERRMODE            => PDO::ERRMODE_EXCEPTION,
            PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC,
            PDO::ATTR_EMULATE_PREPARES   => false,
        ]
    );
} catch (Throwable $e) {
    respondJson(500, [
        'success' => false,
        'error'   => 'db_connection_failed',
        'message' => 'No se pudo conectar a la base de datos.',
        'details' => $e->getMessage(),
    ]);
}

$segments = getPathSegments();
$firstSegment = $segments[0] ?? null;

// Serve documentation files
if ($firstSegment === 'docs') {
    serveDocs($segments);
}

if ($firstSegment === 'install') {
    serveInstaller();
}

$table   = null;
$recordId = null;

if (isset($_GET['table']) && $_GET['table'] !== '') {
    $table = trim((string)$_GET['table']);
} elseif ($firstSegment !== null) {
    $table    = $firstSegment;
    $recordId = $segments[1] ?? null;
}

if ($table === null || $table === '') {
    $schema = describeDatabase($pdo);
    $baseUrl = buildBaseUrl();
    $relativeBase = buildRelativeUrl('/');

    respondJson(200, [
        'success'          => true,
        'message'          => 'API Sofi disponible.',
        'base_url'         => $baseUrl,
        'base_path'        => $relativeBase,
        'available_tables' => array_keys($schema),
        'docs'             => [
            'openapi' => [
                'absolute' => $baseUrl . '/docs/openapi.json',
                'relative' => buildRelativeUrl('/docs/openapi.json'),
            ],
            'swagger' => [
                'absolute' => $baseUrl . '/docs/swagger.json',
                'relative' => buildRelativeUrl('/docs/swagger.json'),
            ],
        ],
        'usage'            => [
            'list'   => 'GET ' . buildRelativeUrl('/{tabla}?columna=valor'),
            'by_id'  => 'GET ' . buildRelativeUrl('/{tabla}/{id}'),
            'create' => 'POST ' . buildRelativeUrl('/{tabla}'),
            'docs'   => 'GET ' . buildRelativeUrl('/docs/openapi.json'),
        ],
    ]);
}

if ($table === 'install') {
    serveInstaller();
}

if (!preg_match('/^[a-zA-Z_][a-zA-Z0-9_]*$/', $table)) {
    respondJson(400, [
        'success' => false,
        'error'   => 'invalid_table',
        'message' => 'Nombre de tabla inválido.',
    ]);
}

ensureManagedTable($pdo, $table);

if (!tableExists($pdo, $table, $db['name'])) {
    respondJson(404, [
        'success' => false,
        'error'   => 'table_not_found',
        'message' => "La tabla {$table} no existe o no está disponible.",
    ]);
}

$metadata = describeTable($pdo, $table);
$method   = $_SERVER['REQUEST_METHOD'] ?? 'GET';

try {
    switch ($method) {
        case 'GET':
            handleGet($pdo, $table, $metadata, $recordId);
            break;
        case 'POST':
            handlePost($pdo, $table, $metadata);
            break;
        default:
            respondJson(405, [
                'success' => false,
                'error'   => 'method_not_allowed',
                'message' => 'Método no soportado. Usa GET o POST.',
            ]);
    }
} catch (Throwable $e) {
    respondJson(500, [
        'success' => false,
        'error'   => 'server_error',
        'message' => 'Ocurrió un error al procesar la solicitud.',
        'details' => $e->getMessage(),
    ]);
}

function describeDatabase(PDO $pdo): array
{
    $tables = [];
    $tableNames = $pdo->query('SHOW TABLES')->fetchAll(PDO::FETCH_COLUMN);

    foreach ($tableNames as $name) {
        $tables[$name] = describeTable($pdo, $name);
    }

    return $tables;
}

function describeTable(PDO $pdo, string $table): array
{
    $columns = $pdo->query("SHOW FULL COLUMNS FROM `$table`")->fetchAll(PDO::FETCH_ASSOC);
    return [
        'columns'    => $columns,
        'primaryKey' => getPrimaryKey($pdo, $table),
    ];
}

function getPathSegments(): array
{
    $uri = parse_url($_SERVER['REQUEST_URI'] ?? '/', PHP_URL_PATH) ?: '/';
    $uri = str_replace('\\', '/', $uri);
    $scriptName = str_replace('\\', '/', $_SERVER['SCRIPT_NAME'] ?? '');
    $scriptDir = rtrim(dirname($scriptName), '/');

    if ($scriptName && str_starts_with($uri, $scriptName)) {
        $uri = substr($uri, strlen($scriptName));
    } elseif ($scriptDir && $scriptDir !== '/' && str_starts_with($uri, $scriptDir)) {
        $uri = substr($uri, strlen($scriptDir));
    }

    $uri = trim($uri, '/');
    return $uri === '' ? [] : explode('/', $uri);
}

function serveDocs(array $segments): void
{
    $docsDir = __DIR__ . '/docs';
    $docPath = $segments[1] ?? null;

    if ($docPath === null || $docPath === '') {
        respondJson(200, [
            'success' => true,
            'docs'    => [
                'openapi' => [
                    'absolute' => buildBaseUrl() . '/docs/openapi.json',
                    'relative' => buildRelativeUrl('/docs/openapi.json'),
                ],
                'swagger' => [
                    'absolute' => buildBaseUrl() . '/docs/swagger.json',
                    'relative' => buildRelativeUrl('/docs/swagger.json'),
                ],
            ],
        ]);
    }

    $requested = implode('/', array_slice($segments, 1));
    $target = realpath($docsDir . '/' . $requested);

    if (!$target || !str_starts_with($target, realpath($docsDir) ?: $docsDir) || !is_file($target)) {
        respondJson(404, [
            'success' => false,
            'error'   => 'not_found',
            'message' => 'Documento no encontrado.',
        ]);
    }

    $ext = strtolower(pathinfo($target, PATHINFO_EXTENSION));
    $mime = match ($ext) {
        'json' => 'application/json; charset=utf-8',
        'yaml', 'yml' => 'application/yaml; charset=utf-8',
        'js'   => 'application/javascript; charset=utf-8',
        'html', 'htm' => 'text/html; charset=utf-8',
        'css'  => 'text/css; charset=utf-8',
        default => 'text/plain; charset=utf-8',
    };

    sendCorsHeaders();
    header('Content-Type: ' . $mime);
    readfile($target);
    exit;
}

function tableExists(PDO $pdo, string $table, ?string $schema = null): bool
{
    static $detectedSchema = null;

    if ($schema === null) {
        if ($detectedSchema === null) {
            $detectedSchema = $pdo->query('SELECT DATABASE()')->fetchColumn();
        }
        $schema = $detectedSchema;
    }

    $stmt = $pdo->prepare(
        'SELECT 1 FROM information_schema.tables WHERE table_schema = :schema AND table_name = :table LIMIT 1'
    );
    $stmt->execute([
        ':schema' => $schema,
        ':table'  => $table,
    ]);

    return (bool) $stmt->fetchColumn();
}

function ensureManagedTable(PDO $pdo, string $table): void
{
    static $checked = [];
    $tableLower = strtolower($table);

    if (isset($checked[$tableLower])) {
        return;
    }

    if (tableExists($pdo, $table)) {
        $checked[$tableLower] = true;
        return;
    }

    $sql = null;
    switch ($tableLower) {
        case 'inspecciones_locales':
            $sql = "CREATE TABLE IF NOT EXISTS `inspecciones_locales` (
                `id` INT NOT NULL AUTO_INCREMENT,
                `local_id` INT NOT NULL,
                `usuario_id` INT NULL,
                `fecha_inicio` DATETIME DEFAULT CURRENT_TIMESTAMP,
                `fecha_fin` DATETIME NULL,
                `estado` ENUM('pendiente','en_progreso','completada') DEFAULT 'pendiente',
                `observaciones` TEXT NULL,
                `created_at` TIMESTAMP NULL DEFAULT CURRENT_TIMESTAMP,
                PRIMARY KEY (`id`),
                KEY `idx_local` (`local_id`)
            ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_0900_ai_ci;";
            break;
        case 'inspecciones_local_equipos':
            $sql = "CREATE TABLE IF NOT EXISTS `inspecciones_local_equipos` (
                `id` INT NOT NULL AUTO_INCREMENT,
                `inspeccion_local_id` INT NOT NULL,
                `inspeccion_id` INT NOT NULL,
                `equipo_id` INT NULL,
                `created_at` TIMESTAMP NULL DEFAULT CURRENT_TIMESTAMP,
                PRIMARY KEY (`id`),
                UNIQUE KEY `uniq_local_inspeccion` (`inspeccion_local_id`, `inspeccion_id`),
                KEY `idx_equipo` (`equipo_id`)
            ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_0900_ai_ci;";
            break;
        default:
            $checked[$tableLower] = true;
            return;
    }

    if ($sql !== null) {
        $pdo->exec($sql);
    }

    $checked[$tableLower] = true;
}

function getPrimaryKey(PDO $pdo, string $table): ?string
{
    $stmt = $pdo->query("SHOW KEYS FROM `$table` WHERE Key_name = 'PRIMARY'");
    $row = $stmt->fetch(PDO::FETCH_ASSOC);
    return $row['Column_name'] ?? null;
}

function handleGet(PDO $pdo, string $table, array $metadata, ?string $recordId): void
{
    try {
        $conditions = [];
        $params     = [];

        if ($recordId !== null && $recordId !== '') {
            $primaryKey = $metadata['primaryKey'];
            if (!$primaryKey) {
                respondJson(400, [
                    'success' => false,
                    'error'   => 'no_primary_key',
                    'message' => "La tabla {$table} no tiene clave primaria definida.",
                ]);
            }
            $conditions[]       = "`$primaryKey` = :pk";
            $params[':pk'] = $recordId;
        }

        foreach ($_GET as $key => $value) {
            if (in_array($key, ['table', 'limit', 'offset', 'sort', 'direction'], true)) {
                continue;
            }
            if (!preg_match('/^[a-zA-Z_][a-zA-Z0-9_]*$/', $key)) {
                continue;
            }
            $placeholder = ':param_' . $key;
            $conditions[] = "`$key` = {$placeholder}";
            $params[$placeholder] = $value;
        }

        $sql = "SELECT * FROM `$table`";
        if ($conditions) {
            $sql .= ' WHERE ' . implode(' AND ', $conditions);
        }

        if (isset($_GET['sort']) && preg_match('/^[a-zA-Z_][a-zA-Z0-9_]*$/', (string) $_GET['sort'])) {
            $direction = strtoupper((string)($_GET['direction'] ?? 'ASC'));
            $direction = in_array($direction, ['ASC', 'DESC'], true) ? $direction : 'ASC';
            $sql .= " ORDER BY `" . $_GET['sort'] . "` {$direction}";
        }

        if (isset($_GET['limit']) && ctype_digit((string) $_GET['limit'])) {
            $limit = max(1, min((int) $_GET['limit'], 1000));
            $sql  .= " LIMIT {$limit}";

            if (isset($_GET['offset']) && ctype_digit((string) $_GET['offset'])) {
                $sql .= " OFFSET " . (int) $_GET['offset'];
            }
        }

        $stmt = $pdo->prepare($sql);
        foreach ($params as $placeholder => $value) {
            $stmt->bindValue($placeholder, $value, getParamType($value));
        }

        $stmt->execute();
        $data = $stmt->fetchAll();

        respondJson(200, [
            'success' => true,
            'data'    => $data,
            'count'   => count($data),
        ]);
    } catch (Throwable $e) {
        respondJson(500, [
            'success' => false,
            'message' => 'Error al ejecutar la consulta.',
            'details' => $e->getMessage(),
        ]);
    }
}

function handlePost(PDO $pdo, string $table, array $metadata): void
{
    $rawInput = file_get_contents('php://input') ?: '';
    $input = json_decode($rawInput, true);

    if ($rawInput === '' && !empty($_POST)) {
        $input = $_POST;
    }

    if (json_last_error() !== JSON_ERROR_NONE) {
        respondJson(400, [
            'success' => false,
            'message' => 'JSON inválido: ' . json_last_error_msg(),
        ]);
    }

    if (!is_array($input) || array_values($input) === $input) {
        respondJson(400, [
            'success' => false,
            'message' => 'Se esperaba un único objeto JSON.',
        ]);
    }

    try {
        $pdo->beginTransaction();
        $primaryKey = $metadata['primaryKey'];

        if (!$primaryKey) {
            throw new RuntimeException("No se encontró primary key para la tabla {$table}.");
        }

        $result = null;

        if (count($input) === 1 && array_key_exists($primaryKey, $input)) {
            $sql = "DELETE FROM `$table` WHERE `$primaryKey` = :id";
            $stmt = $pdo->prepare($sql);
            $stmt->bindValue(':id', $input[$primaryKey], getParamType($input[$primaryKey]));
            $stmt->execute();

            $result = [
                'operation'     => 'delete',
                'affected_rows' => $stmt->rowCount(),
            ];
        } elseif (array_key_exists($primaryKey, $input)) {
            $updates = [];
            $params  = [];

            foreach ($input as $key => $value) {
                if ($key === $primaryKey) {
                    continue;
                }
                if (!sanitizeColumn($key)) {
                    continue;
                }
                $placeholder = ':' . $key;
                $updates[] = "`$key` = {$placeholder}";
                $params[$placeholder] = $value;
            }

            if (!$updates) {
                respondJson(400, [
                    'success' => false,
                    'message' => 'No hay campos válidos para actualizar.',
                ]);
            }

            $params[':pk'] = $input[$primaryKey];
            $sql = "UPDATE `$table` SET " . implode(', ', $updates) . " WHERE `$primaryKey` = :pk";
            $stmt = $pdo->prepare($sql);
            foreach ($params as $placeholder => $value) {
                $stmt->bindValue($placeholder, $value, getParamType($value));
            }
            $stmt->execute();

            $result = [
                'operation'     => 'update',
                'affected_rows' => $stmt->rowCount(),
            ];
        } else {
            $columns      = [];
            $placeholders = [];
            $params       = [];

            foreach ($input as $key => $value) {
                if (!sanitizeColumn($key)) {
                    continue;
                }
                $columns[]      = "`$key`";
                $placeholder    = ':' . $key;
                $placeholders[] = $placeholder;
                $params[$placeholder] = $value;
            }

            if (!$columns) {
                respondJson(400, [
                    'success' => false,
                    'message' => 'No hay columnas válidas para insertar.',
                ]);
            }

            $sql = sprintf(
                'INSERT INTO `%s` (%s) VALUES (%s)',
                $table,
                implode(', ', $columns),
                implode(', ', $placeholders)
            );

            $stmt = $pdo->prepare($sql);
            foreach ($params as $placeholder => $value) {
                $stmt->bindValue($placeholder, $value, getParamType($value));
            }

            $stmt->execute();
            $insertId = $pdo->lastInsertId();
            if ($insertId === '0' || $insertId === '') {
                $insertId = null;
            } elseif (ctype_digit((string) $insertId)) {
                $insertId = (int) $insertId;
            }

            $result = [
                'operation'     => 'insert',
                'insert_id'     => $insertId,
                'affected_rows' => $stmt->rowCount(),
            ];
        }

        $pdo->commit();

        respondJson(200, [
            'success' => true,
            'results' => [$result],
        ]);
    } catch (Throwable $e) {
        if ($pdo->inTransaction()) {
            $pdo->rollBack();
        }

        respondJson(500, [
            'success' => false,
            'message' => $e->getMessage(),
        ]);
    }
}

function sanitizeColumn(string $name): bool
{
    return preg_match('/^[a-zA-Z_][a-zA-Z0-9_]*$/', $name) === 1;
}

function getParamType(mixed $value): int
{
    if (is_int($value)) {
        return PDO::PARAM_INT;
    }
    if (is_bool($value)) {
        return PDO::PARAM_BOOL;
    }
    if ($value === null) {
        return PDO::PARAM_NULL;
    }
    return PDO::PARAM_STR;
}

function getBasePath(): string
{
    $scriptName = $_SERVER['SCRIPT_NAME'] ?? '';
    $dir = str_replace('\\', '/', dirname($scriptName));
    if ($dir === '\\' || $dir === '.') {
        $dir = '';
    }
    if ($dir === '/') {
        $dir = '';
    }
    return rtrim($dir, '/');
}

function buildBaseUrl(): string
{
    $scheme = (!empty($_SERVER['HTTPS']) && $_SERVER['HTTPS'] !== 'off') ? 'https' : 'http';
    $host = $_SERVER['HTTP_HOST'] ?? 'localhost';
    $basePath = getBasePath();
    $url = $scheme . '://' . $host . $basePath;
    return rtrim($url, '/') ?: $url;
}

function buildRelativeUrl(string $path): string
{
    $basePath = getBasePath();
    $normalizedPath = $path === '' ? '' : '/' . ltrim($path, '/');

    $result = $basePath . $normalizedPath;
    if ($result === '') {
        return '/';
    }

    return $result;
}

function serveInstaller(): void
{
    $installer = __DIR__ . '/install/index.php';
    if (is_file($installer)) {
        require $installer;
        exit;
    }

    http_response_code(404);
    echo 'Instalador no disponible.';
    exit;
}

function redirectToInstall(): void
{
    $location = buildBaseUrl() . '/install/';
    header('Location: ' . $location);
    exit;
}

function sendCorsHeaders(): void
{
    header('Access-Control-Allow-Origin: *');
    header('Access-Control-Allow-Methods: GET, POST, OPTIONS');
    header('Access-Control-Allow-Headers: Content-Type, Authorization, Accept');
}

function respondJson(int $status, array $payload): void
{
    sendCorsHeaders();
    header('Content-Type: application/json; charset=utf-8');
    http_response_code($status);
    echo json_encode($payload, JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES);
    exit;
}
